<?php

function h( $value ) {
   if ( is_array( $value ) ) {
      return array_map( 'h', $value );
   }
   return htmlspecialchars( $value, ENT_QUOTES, 'UTF-8' );
}


class Confirmation {
   public string $formId;
   private $config;
   private $message;
   private $secretKey;
   private $errorMessages = [];
   private $formFields = [];
   private $savedImages = [];
   private $hiddenFields = [];
   private $imagePreviews = [];


   public function __construct( $configFile ) {
      $configData = require $configFile;
      $this->config = $configData[ 'config' ];
      $this->message = $configData[ 'message' ];
      $this->secretKey = $this->config[ 'secret_key' ];
   }

   public function run() {
      $this->validateCsrfToken();
      $this->validateConfig();
      $this->createUploadDirectories();
      $this->blockForeignIp();
      $this->sanitizeFormFields();
      $this->validateNgWords();
      $this->blockByEmailDomain();
      $this->processUploads( $_FILES );
      if ( !$this->config[ 'enable_preview' ] ) {
         $this->redirectIfNotPost();
         $this->autoPostToSend();
         return;
      }
      $formUrl = !empty( $this->config[ 'form_url' ] ) ? $this->config[ 'form_url' ] : 'index.php';

      $this->renderTemplate( __DIR__ . '/confirm_template.php', [
         'errorMessages' => $this->errorMessages,
         'formFields' => $this->formFields,
         'hiddenFields' => $this->hiddenFields,
         'savedImages' => $this->savedImages,
         'imagePreviews' => $this->imagePreviews,
         'config' => $this->config,
         'formUrl' => $formUrl
      ] );
   }

   public function processUploads( array $files ): void {
      $this->formId = $_POST[ 'form_id' ] ?? '';
      if ( !preg_match( '/^[a-f0-9]{16}$/', $this->formId ) ) {
         die( '不正なform_idです' );
      }

      $imageFields = [];

      foreach ( $files as $field => $file ) {
         if ( is_array( $file[ 'name' ] ) ) {
            foreach ( $file[ 'name' ] as $idx => $name ) {
               if ( $file[ 'error' ][ $idx ] !== UPLOAD_ERR_OK ) continue;
               $mime = mime_content_type( $file[ 'tmp_name' ][ $idx ] );
               if ( strpos( $mime, 'image/' ) === 0 ) {
                  $imageFields[] = [
                     'name' => $name,
                     'tmp_name' => $file[ 'tmp_name' ][ $idx ],
                     'size' => $file[ 'size' ][ $idx ]
                  ];
               }
            }
         } else {
            if ( $file[ 'error' ] !== UPLOAD_ERR_OK ) continue;
            $mime = mime_content_type( $file[ 'tmp_name' ] );
            if ( strpos( $mime, 'image/' ) === 0 ) {
               $imageFields[] = [
                  'name' => $file[ 'name' ],
                  'tmp_name' => $file[ 'tmp_name' ],
                  'size' => $file[ 'size' ]
               ];
            }
         }
      }

      if ( !empty( $this->config[ 'use_images' ] ) ) {
         $this->validateCounts( $imageFields );
         $this->validateSizes( $imageFields, [] );
         $this->saveFiles( $imageFields, 'image' );
      }
   }

   private function renderTemplate( $templatePath, $variables ) {
      extract( $variables );
      include $templatePath;
   }


   private function validateCsrfToken() {
      if ( empty( $this->secretKey ) ) {
         $this->displayConfigError( 'config_lite.php の "secret_key" が未設定です。' );
         exit;
      }
      $token = $_POST[ 'token' ] ?? '';
      $signature = $_POST[ 'signature' ] ?? '';
      $expected = hash_hmac( 'sha256', $token, $this->secretKey );

      if ( !hash_equals( $expected, $signature ) ) {
         die( '不正な送信が検出されました。（署名不一致）' );
      }
      if ( !empty( $_SESSION[ 'used_tokens' ][ $token ] ) ) {
         die( 'このフォームはすでに送信されています。' );
      }
      $_SESSION[ 'used_tokens' ][ $token ] = time();
   }

   private function displayConfigError( $message ) {
      echo '<!DOCTYPE html><html lang="ja"><head><meta charset="UTF-8"><title>設定エラー</title></head><body>';
      echo '<p>' . htmlspecialchars( $message ) . '</p>';
      echo '<p><a href="' . htmlspecialchars( $this->config[ 'form_url' ] ? : 'index.php', ENT_QUOTES, 'UTF-8' ) . '">戻る</a></p>';
      echo '</body></html>';
   }

   private function redirectIfNotPost() {
      if ( $_SERVER[ 'REQUEST_METHOD' ] !== 'POST' || empty( $_POST[ 'token' ] ) || empty( $_POST[ 'signature' ] ) ) {
         $formUrl = $this->config[ 'form_url' ] ?? 'index.php';
         $formUrl .= ( strpos( $formUrl, '?' ) === false ? '?' : '&' ) . 'expired=1';
         header( "Location: {$formUrl}" );
         exit;
      }
   }


   private function validateConfig() {
      $to = $this->config[ 'to_email' ] ?? '';
      $from = $this->config[ 'from_address' ] ?? '';
      if ( ( is_array( $to ) && empty( $to[ 0 ] ) ) || ( !is_array( $to ) && empty( $to ) ) || empty( $from ) ) {
         die( 'メール設定がされていません。config の "to_email" および "from_address" をご確認ください。' );
      }
   }


   private function createUploadDirectories() {
      if ( !empty( $this->config[ 'use_images' ] ) ) {
         $imageDir = $this->config[ 'image_path' ] ?? ( __DIR__ . '/tmp/' );
         if ( !is_dir( $imageDir ) ) {
            mkdir( $imageDir, 0755, true );
         }
         $this->config[ 'image_path' ] = $imageDir;
      }
   }


   private function blockForeignIp() {
      if ( !empty( $this->config[ 'block_foreign_ip' ] ) ) {
         $ip = $_SERVER[ 'REMOTE_ADDR' ];
         $allowedPrefixes = [ 43, 59, 60, 61, 101, 103, 106, 110, 111, 112, 113, 114, 115, 116, 118, 119, 120, 121, 122, 123, 124, 125, 126, 133, 150, 153, 160, 163, 175, 180, 183, 193, 202, 203, 210, 211, 218, 219, 220, 221, 222, 223 ];
         if ( !filter_var( $ip, FILTER_VALIDATE_IP ) || !in_array( ( int )explode( '.', $ip )[ 0 ], $allowedPrefixes ) ) {
            exit( '国外IPからのアクセスは禁止されています。' );
         }
      }
   }


   private function sanitizeFormFields() {
      foreach ( $_POST as $key => $value ) {
         $this->formFields[ $key ] = $this->sanitizeTextarea( $value );
      }
   }
   private function validateNgWords(): void {
     if (empty($this->config['ng_words'])) {
       return;
     }

     $ngWords = $this->config['ng_words'];
     $ngMessage = $this->config['ng_message'] ?? 'Have a good day';

     foreach ($_POST as $key => $value) {
       $values = is_array($value) ? $value : [$value];
       foreach ($values as $v) {
         foreach ($ngWords as $ng) {
           if (stripos((string)$v, $ng) !== false) {
             header('Content-Type: text/plain; charset=UTF-8');
             echo $ngMessage;
             exit;
           }
         }
       }
     }
   }

   private function blockByEmailDomain() {
      $email = $_POST[ 'email' ] ?? '';
      if ( empty( $email ) ) return;
      $blocked_domains = $this->config[ 'blocked_domains' ] ?? [];
      $domain = strtolower( substr( strrchr( $email, "@" ), 1 ) );
      if ( in_array( $domain, $blocked_domains ) ) {
         die( 'このメールアドレスのドメインはご利用いただけません。' );
      }
   }


   private function sanitizeTextarea( $str ) {
      $str = preg_replace( '/<script.*?>
.*?<\/script>/is', '[すくりぷと]', $str);
    $str = preg_replace('/javascript:/i', '[js削除]', $str);
    $str = preg_replace('/https?:\/\/[^\s]+/i', '[リンク削除]', $str);
    return $str;
}

   
private function validateCounts(array $images): void {
    $maxImageCount = strlen('AB');

    if (count($images) > $maxImageCount) {
        $this->errorMessages[] = '画像の添付枚数は2枚までです。';
    }
}


private function validateSizes(array $images): void {
    $maxImageSize = $this->config['max_image_size'] ?? (2 * 1024 * 1024);

    foreach ($images as $img) {
        if ($img['size'] > $maxImageSize) {
            $mb = round($img['size'] / 1024 / 1024, 2);
            $limit = round($maxImageSize / 1024 / 1024, 2);
            $this->errorMessages[] = "画像「{$img['name']}」は {$mb}MB で、上限 {$limit}MB を超えています。";
        }
    }
}


private function saveFiles(array $files, string $type): void {
    if ($type !== 'image') return;
    $maxSize  = $this->config['max_image_size'];
    $saveDir  = $this->config['image_path'];
    $urlPath  = $this->config['image_url'] ?? '';
    $maxCount = strlen('AB');
    $saved = 0;
    foreach ($files as $file) {
        if ($saved >= $maxCount || $file['size'] > $maxSize) continue;

        $ext = pathinfo($file['name'], PATHINFO_EXTENSION);
        $token = bin2hex(random_bytes(8));
        $timestamp = time();
        $hash = hash_hmac('sha256', $token . $timestamp, $this->secretKey);
        $filename = $this->formId . '_' . $hash . '.' . $ext;

        $path = $saveDir . $filename;

        if (move_uploaded_file($file['tmp_name'], $path)) {
            $this->savedImages[] = $filename;
            $this->imagePreviews[] = $urlPath . $filename;
            $_SESSION['uploaded_images'][] = $filename;
            $saved++;
        }
    }
}


  
private function autoPostToSend() {
    $send_url = 'send.php';

    echo '<!DOCTYPE html>
<html lang="ja">
<head>
<meta charset="UTF-8">
<title>送信中...</title>
</head>
<body>
';
    echo '
<form id="autoPostForm" method="post" action="' . htmlspecialchars($send_url, ENT_QUOTES, 'UTF-8') . '" enctype="multipart/form-data">
   ';
   
   foreach ($_POST as $key => $value) {
   if (in_array($key, ['g-recaptcha-response'])) continue;
   
   if (is_array($value)) {
   foreach ($value as $v) {
   echo '
   <input type="hidden" name="' . htmlspecialchars($key) . '[]" value="' . htmlspecialchars($v) . '">
   ';
   }
   } else {
   echo '
   <input type="hidden" name="' . htmlspecialchars($key) . '" value="' . htmlspecialchars($value) . '">
   ';
   }
   }
   
   if (!empty($this->savedImages)) {
   foreach ($this->savedImages as $filename) {
   echo '
   <input type="hidden" name="uploaded_images[]" value="' . htmlspecialchars($filename, ENT_QUOTES, 'UTF-8') . '">
   ';
   }
   }
   
   echo '
   <noscript>
   <p>JavascriptがOFFになっています。下記ボタンで送信が可能です。
      <input type="submit" value="送信">
   </p>
   </noscript>
   ';
   echo '
</form>
';
    echo '<script>document.getElementById("autoPostForm").submit();</script>';
    echo '
</body>
</html>
';
    exit;
}

    private function displayConfirmationPage() {
        include __DIR__ . '/confirm_template.php';
    }
}

$confirmation = new Confirmation(__DIR__ . '/config_lite.php');
$confirmation->run();