<?php
session_start();
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die('送信内容に不整合がありました。再度フォームからご送信ください。');
}
class SendHandler {
   private $config;
   private $message;
   private $secretKey;
   private $formFieldsAdmin = [];
   private $formFieldsUser = [];
   private $uploadedImages = [];
   private $userNotes = [];
   private $adminSuccess = true;
   private $userSuccess = true;
   private $sendMessageAdmin = '';


   public function __construct( $configFile ) {
      $configData = require $configFile;
      $this->config = $configData[ 'config' ];
      $this->message = $configData[ 'message' ];
      $this->secretKey = $this->config[ 'secret_key' ];
   }

   public function run() {
      $this->redirectIfNotPost();
      $this->validateCsrfToken();
      $this->validateRequiredFields();
      $this->validateRecaptcha();
      $this->validateFormContents();
      $this->sanitizeFormData();
      $this->validateAttachmentsLightly();
      $this->checkRequiredConfig();
      $this->prepareEmailData();
      $this->sendAdminEmail();
      $this->sendUserEmail();
      $this->cleanupUploadedFiles();
      $this->handleRedirectOrDisplayResult();
   }

   private function validateCsrfToken() {
      if ( empty( $this->secretKey ) ) {
         $this->displayConfigError( 'config.php の "secret_key" が未設定です。' );
         exit;
      }
      $token = $_POST[ 'token' ] ?? '';
      $signature = $_POST[ 'signature' ] ?? '';
      $expected = hash_hmac( 'sha256', $token, $this->secretKey );

      if ( !hash_equals( $expected, $signature ) ) {
         die( '不正な送信が検出されました。（署名不一致）' );
      }

      if ( !empty( $_SESSION[ 'used_tokens' ][ $token ] ) ) {
         die( 'このフォームはすでに送信されています。再送信できません。' );
      }
      $_SESSION[ 'used_tokens' ][ $token ] = time();
   }

   private function validateRequiredFields() {
      $required = $this->config[ 'require_fields' ] ?? [];

      foreach ( $required as $field ) {
         if ( !isset( $_POST[ $field ] ) || trim( $_POST[ $field ] ) === '' ) {
            $this->displaySendError( "【{$field}】は必須項目です。入力してください。" );
            exit;
         }
      }
   }

   private function checkRequiredConfig() {
      $requiredKeys = [
         'image_path',
         'image_url'
      ];

      foreach ( $requiredKeys as $key ) {
         if ( !array_key_exists( $key, $this->config ) ) {
            $this->displaySendError( "config_lite.php の設定「{$key}」が存在しません。設定を追加してください。" );
            exit;
         }

         if ( $this->config[ $key ] === '' || $this->config[ $key ] === null ) {
            $this->displaySendError( "config_lite.php の設定「{$key}」が未入力です。正しい値を設定してください。" );
            exit;
         }
      }
   }


   private function displaySendError( string $message ) {
      echo '<!DOCTYPE html><html lang="ja"><head><meta charset="UTF-8"><title>送信エラー</title></head><body>';
      echo '<p>' . htmlspecialchars( $message, ENT_QUOTES, 'UTF-8' ) . '</p>';
      echo '<p><a href="' . htmlspecialchars( $this->config[ 'form_url' ] ? : 'index.php' ) . '">戻る</a></p>';
      echo '</body></html>';
   }


   private function redirectIfNotPost() {
      if ( $_SERVER[ 'REQUEST_METHOD' ] !== 'POST' ) {
         $form_url = $config[ 'form_url' ] !== '' ? $config[ 'form_url' ] : './';
         header( "Location: {$formUrl}" );
         exit;
      }
   }

   private function validateRecaptcha() {
      if ( isset( $_POST[ 'g-recaptcha-response' ] ) ) {
         if ( empty( $this->config[ 'recaptcha_secret' ] ) ) {
            exit( 'reCAPTCHAが有効化されていますが、シークレットキーが設定されていません。' );
         }

         $recaptchaSecret = $this->config[ 'recaptcha_secret' ];
         $recaptchaResponse = $_POST[ 'g-recaptcha-response' ];

         $verify = file_get_contents( "https://www.google.com/recaptcha/api/siteverify?secret={$recaptchaSecret}&response={$recaptchaResponse}&remoteip=" . $_SERVER[ 'REMOTE_ADDR' ] );
         $responseData = json_decode( $verify );

         if ( empty( $responseData->success ) ) {
            exit( 'reCAPTCHAの認証に失敗しました。' );
         }
      }

   }

   private function validateFormContents() {
      foreach ( $_POST as $key => $value ) {
         $values = is_array( $value ) ? $value : [ $value ];
         foreach ( $values as $v ) {
            if ( preg_match( '/https?:\/\/[^\s]+/i', $v ) ) {
               die( "リンクを含む入力は許可されていません。" );
            }
         }
      }
   }


   private function sanitizeFormData() {
      $excludedKeys = [ 'token', 'signature', 'g-recaptcha-response', 'form_id', 'uploaded_images' ];
      foreach ( $_POST as $key => $value ) {
         if ( !in_array( $key, $excludedKeys, true ) ) {
            $this->formFieldsAdmin[ $key ] = $this->sanitizeTextarea( $value );
            if ( $key !== 'hidden' && $key !== 'hidden2' ) {
               $this->formFieldsUser[ $key ] = $this->sanitizeTextarea( $value );
            }
         }
      }
      $this->uploadedImages = $_POST[ 'uploaded_images' ] ?? [];
   }

   private function sanitizeTextarea( $str ) {
      if ( is_array( $str ) ) {
         return array_map( [ $this, 'sanitizeTextarea' ], $str );
      }
      return htmlspecialchars( $str, ENT_QUOTES, 'UTF-8' );
   }


   private function validateAttachmentsLightly() {
      foreach ( $this->uploadedImages as $filename ) {
         $filepath = $this->config[ 'image_path' ] . basename( $filename );
         if ( !file_exists( $filepath ) ) {
            $this->displaySendError( "画像ファイルが見つかりません。" );
            exit;
         }
      }
   }

   private function prepareEmailData() {
      if ( !empty( $this->uploadedImages ) ) {
         $this->userNotes[] = '※画像を添付しました';
      }
   }

   private function appendServerInfo(): string {
      $server = $_SERVER;
      $EOL = "\r\n"; // Outlookなどのメーラー対策
      $lines = "-----------------------------------------------------" . $EOL;
      $lines .= "[送信日時] " . date( "Y年m月d日(D) H時i分s秒" ) . $EOL;
      $lines .= "[IPアドレス] " . ( $server[ 'REMOTE_ADDR' ] ?? '不明' ) . $EOL;
      $lines .= "[ホスト] " . ( gethostbyaddr( $server[ 'REMOTE_ADDR' ] ?? '' ) ? : '不明' ) . $EOL;
      $lines .= "[USER AGENT] " . ( $server[ 'HTTP_USER_AGENT' ] ?? '不明' ) . $EOL;
      $lines .= "-----------------------------------------------------" . $EOL;

      return $lines;
   }


   private function sendAdminEmail() {
      if ( !empty( $this->config[ 'auto_reply_admin' ] ) ) {
         $subject_raw = $this->config[ 'admin_mail_subject' ] ?? '【お問い合わせ】フォームからの送信';
         $subject = mb_encode_mimeheader( $subject_raw, 'ISO-2022-JP-MS', 'UTF-8' );
         $to_email_raw = $this->config[ 'to_email' ] ?? '';
         if ( !is_string( $to_email_raw ) || strpos( $to_email_raw, ',' ) !== false ) {
            die( 'メールアドレスは1件のみ指定可能です。' );
         }
         $adminName = $this->config[ 'admin_name' ] ?? '';
         $to = mb_encode_mimeheader( $adminName, 'ISO-2022-JP-MS', 'UTF-8' ) . ' <' . trim( $to_email_raw ) . '>';
         $from_email = $this->config[ 'from_address' ];
         $reply_email = $_POST[ 'email' ] ?? $to_email_raw;
         $headers = "From: {$from_email}\r\nReply-To: {$reply_email}\r\n";
         $body = $this->createEmailBody( $this->formFieldsAdmin );
         $body .= "\n" . $this->appendServerInfo();
         $files = $this->collectUploadedFiles( $this->uploadedImages, $this->config[ 'image_path' ] );
         if ( !empty( $this->config[ 'admin_footer_note' ] ) ) {
            $body .= "\n" . $this->config[ 'admin_footer_note' ] . "\n";
         }
         $this->adminSuccess = $this->sendMailWithAttachments(
            $to,
            $subject,
            $body,
            $headers,
            $files
         );
      }
   }

   private function sendUserEmail() {
      $email = $_POST[ 'email' ] ?? '';
      if ( empty( $email ) ) {
         $this->displaySendError( "メールアドレスが未入力です。" );
         exit;
      }
      if ( !empty( $this->config[ 'auto_reply_user' ] ) ) {
         $subject_raw = $this->config[ 'user_mail_subject' ] ?? '【自動返信】お問い合わせありがとうございます';
         $subject = mb_encode_mimeheader( $subject_raw, 'ISO-2022-JP-MS', 'UTF-8' );
         $from_name = $this->config[ 'from_name' ] ?? 'お問い合わせ窓口';
         $from_email = $this->config[ 'from_address' ];
         $from_header = '=?UTF-8?B?' . base64_encode( $from_name ) . '?= <' . $from_email . '>';
         $reply_to = $this->config[ 'to_email' ] ?? '';
         if ( !is_string( $reply_to ) || strpos( $reply_to, ',' ) !== false ) {
            die( 'config の "to_email" は1件のメールアドレスのみ指定してください。' );
         }
         $headers = "From: {$from_header}\r\nReply-To: {$reply_to}\r\n";
         $body = $this->createEmailBody( $this->formFieldsUser, true );
         foreach ( $this->userNotes as $note ) {
            $body .= "\n" . ( is_array( $note ) ? print_r( $note, true ) : $note ) . "\n";
         }
         $files = [];
         if ( !empty( $this->config[ 'attach_images_to_user_mail' ] ) ) {
            $files = $this->collectUploadedFiles( $this->uploadedImages, $this->config[ 'image_path' ] );
         }
         $this->userSuccess = $this->sendMailWithAttachments(
            $email,
            $subject,
            $body,
            $headers,
            $files
         );
      }
   }


   private function createEmailBody( $formFields, $forUser = false ) {
      $body = "";
      if ( $forUser && !empty( $this->message[ 'return_header' ] ) ) {
         $onamae = $formFields[ 'お名前' ] ?? 'ユーザー';
         $header = str_replace( '{{onamae}}', $onamae, $this->message[ 'return_header' ] );
         $body .= $header . "\n\n";
      }

      foreach ( $formFields as $key => $value ) {
         $body .= "【{$key}】" . ( is_array( $value ) ? implode( ', ', $value ) : $value ) . "\n";
      }

      if ( $forUser && !empty( $this->message[ 'return_footer' ] ) ) {
         $body .= "\n" . $this->message[ 'return_footer' ] . "\n";
      }
      return $body;
   }


   private function collectUploadedFiles( $uploadedFiles, $path ) {
      $files = [];
      foreach ( $uploadedFiles as $name ) {
         $filepath = $path . basename( $name );
         if ( file_exists( $filepath ) ) {
            $files[] = $filepath;
         }
      }
      return $files;
   }

   private function sendMailWithAttachments( $to, $subject, $message, $headers, $files = [] ) {
      $boundary = "==Multipart_" . md5( uniqid() );
      $headers .= "MIME-Version: 1.0\r\n";
      $headers .= "Content-Type: multipart/mixed; boundary=\"{$boundary}\"\r\n";

      $body = "--{$boundary}\r\n";
      $body .= "Content-Type: text/plain; charset=\"UTF-8\"\r\n";
      $body .= "Content-Transfer-Encoding: 7bit\r\n\r\n";
      $body .= $message . "\r\n\r\n";

      foreach ( $files as $filepath ) {
         if ( !file_exists( $filepath ) ) {
            continue;
         }
         $filename = basename( $filepath );
         $mime = mime_content_type( $filepath );
         $content = chunk_split( base64_encode( file_get_contents( $filepath ) ) );

         $body .= "--{$boundary}\r\n";
         $body .= "Content-Type: {$mime}; name=\"{$filename}\"\r\n";
         $body .= "Content-Disposition: attachment; filename=\"{$filename}\"\r\n";
         $body .= "Content-Transfer-Encoding: base64\r\n\r\n";
         $body .= $content . "\r\n\r\n";
      }

      $body .= "--{$boundary}--";
      return mail( $to, $subject, $body, $headers );
   }


   private function cleanupUploadedFiles() {
      $form_id = $_POST[ 'form_id' ] ?? '';
      if ( !preg_match( '/^[a-f0-9]{16}$/', $form_id ) ) {
         return;
      }
      if ( !$this->config[ 'save_images' ] && !empty( $_POST[ 'uploaded_images' ] ) ) {
         foreach ( $_POST[ 'uploaded_images' ] as $name ) {
            if ( strpos( $name, $form_id . '_' ) !== 0 ) continue;
            $path = $this->config[ 'image_path' ] . basename( $name );
            if ( file_exists( $path ) )unlink( $path );
         }
      }
   }


   private function handleRedirectOrDisplayResult() {
      $sendSuccess = $this->adminSuccess && $this->userSuccess;
      if ( !empty( $this->config[ 'redirect_url' ] ) ) {
         $redirectUrl = $this->config[ 'redirect_url' ];
         $redirectUrl .= ( strpos( $redirectUrl, '?' ) === false ? '?' : '&' ) . ( $sendSuccess ? 'send=ok' : 'send=fail' );
         header( "Location: {$redirectUrl}" );
         exit;
      }
      $formUrl = !empty( $this->config[ 'form_url' ] ) ? $this->config[ 'form_url' ] : 'index.php';

      echo '<!DOCTYPE html><html lang="ja"><head><meta charset="UTF-8"><title>送信完了</title>';
      echo '</head><body>';

      if ( $sendSuccess ) {
         echo '<p>送信が完了しました。ありがとうございました。</p>';
      } else {
         echo '<p style="color:red;">送信に失敗しました。恐れ入りますが、時間を置いて再度お試しください。</p>';
      }
      echo '<p><a href="' . htmlspecialchars( $formUrl, ENT_QUOTES, 'UTF-8' ) . '">戻る</a></p>';
      echo '</body></html>';
      exit;
   }

}

// 実行
$sendHandler = new SendHandler( __DIR__ . '/config_lite.php' );
$sendHandler->run();